<?php
/**
 * Plugin Name: Custom Job API Endpoint
 * Description: Adds a custom REST API endpoint for posting new jobs.
 * Author: Your Name
 * Version: 1.0
 */

// This function registers a custom REST API endpoint.
function register_job_api_route() {
    register_rest_route('custom/v1', '/jobs', array(
        'methods' => 'POST',
        'callback' => 'create_job_post',
        'permission_callback' => 'check_permission'
    ));
}

add_action('rest_api_init', 'register_job_api_route');

/**
 * Checks if the user has permission to post to the API.
 * This is where you would place your authentication logic.
 * For this example, we check for a valid Application Password.
 *
 * @return bool
 */
function check_permission() {
    return current_user_can('publish_posts');
}

/**
 * Creates a new job post from the API request.
 *
 * @param WP_REST_Request $request The REST API request object.
 * @return WP_REST_Response|WP_Error
 */
function create_job_post($request) {
    // Get the parameters from the request body.
    $title = $request->get_param('title');
    $content = $request->get_param('description');

    // Basic validation to ensure required fields are present.
    if (empty($title) || empty($content)) {
        return new WP_Error('empty_fields', 'Title and description are required.', array('status' => 400));
    }

    // Prepare the new post data.
    $post_data = array(
        'post_title'    => sanitize_text_field($title),
        'post_content'  => wp_kses_post($content),
        'post_status'   => 'publish',
        'post_type'     => 'post', // Change this to 'job' or another custom post type if it exists.
        'post_author'   => 1 // You may want to change the author ID.
    );

    // Insert the post into the database.
    $post_id = wp_insert_post($post_data);

    // Check for errors during post creation.
    if (is_wp_error($post_id)) {
        return new WP_Error('post_creation_failed', 'Failed to create post.', array('status' => 500));
    }

    // Return a success response with the ID of the new post.
    return new WP_REST_Response(array('message' => 'Job posted successfully!', 'post_id' => $post_id), 201);
}
